#include <stdio.h>
#include <string.h>

// start addr of DAB firmware in external FLASH
#define		FLASH_START_ADDR_DAB		0x6000			// DAB+ Shield
//#define		FLASH_START_ADDR_DAB		0x86000		// DAB+ Radio

#define DAB_MAX_SERVICES		16		// max number of DAB servcies

// service 
typedef struct _dab_services
{
  uint8_t   freq;
  uint32_t  serviceID;
  uint32_t  compID;
  char      label[17];
} dab_service_t;

// variables from si468x function
extern unsigned char 	spi_buf[];
extern uint8_t				cmd_error;
extern const uint32_t	dab_freq[];
extern uint32_t 			dabfreq;
extern uint8_t   			ver_major;
extern uint8_t   			ver_minor;
extern uint8_t   			ver_build;
extern uint8_t   			chip_rev;
extern uint8_t   			rom_id;
extern uint16_t  			part_no;
extern uint16_t 			audio_bit_rate;			// audio bit rate of the current audio service (kbps)
extern uint16_t 			audio_sample_rate;	// Sample rate of the current audio service (Hz)
extern uint8_t				audio_mode;					// audio_mode 0: dual,1: mono,2: stereo,3: joint stereo
extern bool						dabplus;						// DAB+
extern uint8_t 				service_mode;				// Indicates the service mode of the sub-channel 
																					// e.g. 4: DAB+,5: DAB                          
// variables from FM function
extern uint8_t		dab_status;
extern uint16_t		value;
extern bool				valid;
extern char 			service_data[];

// internal variables
uint8_t				freq_index;									// DAB frequency table index 
char 					ensemble_name[17];					// DAB ensemble name
uint8_t				numberofservices = 0;				// DAB max number of services
dab_service_t	service[DAB_MAX_SERVICES];	// DAB service details

//-----------------------------------------------------------------------------------------
// initialize DAB mode
// return  status
//-----------------------------------------------------------------------------------------
uint8_t dab_init()
{
	if (dab_status = si468x_reset() != 0) return(dab_status);
	if (dab_status = si468x_load_init() != 0) return(dab_status);
	if (dab_status = si468x_host_load() != 0) return(dab_status);
	if (dab_status = si468x_load_init() != 0) return(dab_status);
	if (dab_status = si468x_flash_set_property(0x0001, 5000) != 0) return(dab_status);
	if (dab_status = si468x_flash_load(FLASH_START_ADDR_DAB) != 0) return(dab_status);
	if (dab_status = si468x_boot(&value) != 0) return(dab_status);
	if (dab_status = si468x_dab_set_freq_list() != 0) return(dab_status);	
	if (dab_status = si468x_set_property(0x0000, 0x0010) != 0) return(dab_status); 	// INT_CTL_ENABLE  DSRVIEN
	if (dab_status = si468x_set_property(0x1710, 0xF9FF) != 0) return(dab_status);  // DAB_TUNE_FE_VARM
	if (dab_status = si468x_set_property(0x1711, 0x0172) != 0) return(dab_status);  // DAB_TUNE_FE_VARB
	if (dab_status = si468x_set_property(0x1712, 0x0001) != 0) return(dab_status);  // DAB_TUNE_FE_CFG
	if (dab_status = si468x_set_property(0x8100, 0x0001) != 0) return(dab_status);  // DIGITAL_SERVICE_INT_SOURCE DSRVPCKTINT
	if (dab_status = si468x_set_property(0xb400, 0x0007) != 0) return(dab_status);  // DAB_XPAD_ENABLE
	return(dab_status);
}

//-----------------------------------------------------------------------------------------
// Tune to DAB frequency and get enseble infos
// freq_idx  Index of DAB frequency table
//-----------------------------------------------------------------------------------------
void dab_tune(uint8_t freq_idx)
{
	numberofservices = 0;
	freq_index = freq_idx;
	service_data[0] = '\0';
	ensemble_name[0] = '\0';
	si468x_dab_tune_freq(freq_idx);
	dab_get_ensemble_info();
}

//-----------------------------------------------------------------------------------------
// Waiting until service list is received
//-----------------------------------------------------------------------------------------
void dab_parse_service_list(void)
{
	uint16_t i, j, offset;

	uint32_t serviceID;
	uint32_t componentID;
	uint16_t numberofcomponents;

	uint16_t listsize;
	uint16_t version;

	listsize = spi_buf[5] + (spi_buf[6] << 8);
	version = spi_buf[7] + (spi_buf[8] << 8);
	(void)listsize; //not used
	(void)version;	//not used
	
	numberofservices = spi_buf[9];
	if(numberofservices > DAB_MAX_SERVICES)
	{
		numberofservices = DAB_MAX_SERVICES;
	}

	offset = 13;

	for (i = 0; i < numberofservices; i++)
	{
		serviceID = spi_buf[offset + 3];
		serviceID <<= 8;
		serviceID += spi_buf[offset + 2];
		serviceID <<= 8;
		serviceID += spi_buf[offset + 1];
		serviceID <<= 8;
		serviceID += spi_buf[offset];
		componentID = 0;

		numberofcomponents = spi_buf[offset + 5] & 0x0F;

		for (j = 0; j < 16; j++)
		{
			service[i].label[j] = spi_buf[offset + 8 + j];
		}
		service[i].label[16] = '\0';

		offset += 24;

		for (j = 0; j < numberofcomponents; j++)
		{
			if (j == 0)
			{
				componentID = spi_buf[offset + 3];
				componentID <<= 8;
				componentID += spi_buf[offset + 2];
				componentID <<= 8;
				componentID += spi_buf[offset + 1];
				componentID <<= 8;
				componentID += spi_buf[offset];
			}
			offset += 4;
		}

		service[i].serviceID = serviceID;
		service[i].compID = componentID;
	}
}

//-----------------------------------------------------------------------------------------
// Checks if actual Service is valid
// service valid true/false
//-----------------------------------------------------------------------------------------
static bool dab_service_valid(void)
{
  si468x_dab_digrad_status();
  si468x_responseN(23);

  if (((spi_buf[6] & 0x01) == 0x01) && (spi_buf[7] > 0x20) && (spi_buf[9] > 25))
  {
    return true;
  }
  else
  {
    return false;
  }
}

//-----------------------------------------------------------------------------------------
// Waiting until service list is received
//-----------------------------------------------------------------------------------------
void dab_wait_service_list(void)
{
	uint32_t	timeout;
	timeout = 1000;
	do
	{
		delay(4);
		si468x_dab_get_event_status();
		si468x_responseN(8);
		timeout--;
		if(timeout == 0)
		{
			cmd_error |= 0x80;
			break;
		}
	}
	while ((spi_buf[6] & 0x01) == 0x00); //Service List Ready ?
}

//-----------------------------------------------------------------------------------------
// set service
//-----------------------------------------------------------------------------------------
void dab_set_service(uint8_t index)
{
	char buffer[16];

	si468x_start_digital_service(service[index].serviceID, service[index].compID);
	si468x_dab_get_subchan_info(service[index].serviceID, service[index].compID);
	si468x_dab_get_audio_info();
	Serial.print(service[index].label);
  Serial.print(F(" AM="));
  sprintf(buffer, "%u", audio_mode);
  Serial.print(buffer);
  Serial.print(F(" BR="));
  sprintf(buffer, "%u", audio_bit_rate);
  Serial.print(buffer);
  Serial.print(F(" SR="));
  sprintf(buffer, "%u", audio_sample_rate);
  Serial.print(buffer);
  Serial.print(F(" SM="));
  sprintf(buffer, "%u\n", service_mode);
  Serial.print(buffer);
}

//-----------------------------------------------------------------------------------------
// Get current ensemble info
//-----------------------------------------------------------------------------------------
void dab_get_ensemble_info(void)
{
	uint16_t len;
	uint8_t i;

	si468x_dab_digrad_status();
	si468x_responseN(23);
	dabfreq = spi_buf[13] + ((uint32_t)spi_buf[14] << 8) + ((uint32_t)spi_buf[15] << 16) + ((uint32_t)spi_buf[16] << 24);
  rssi = spi_buf[7];
  snr = spi_buf[8];
	if (dab_service_valid() == true)
	{
		dab_wait_service_list();
		si468x_get_ensemble_info();
		si468x_responseN(29);
		for (i = 0; i < 16; i++)
		{
			ensemble_name[i] = ((char)spi_buf[7 + i]);
		}
		ensemble_name[16] = '\0';
		si468x_get_digital_service_list();
		si468x_responseN(6);
		len = spi_buf[5] + (spi_buf[6] << 8);
		si468x_responseN(len + 4);
		dab_parse_service_list();
	}
	else
	{
		//No services
	}
}

//-----------------------------------------------------------------------------------------
// Scan DAB frequencies and report valid ensables 
//-----------------------------------------------------------------------------------------
void dab_scan(void)
{
  uint8_t freq_index;
  char buffer[16];

  for (freq_index = 0; freq_index < si468x_max_freq_index(); freq_index++)
  {
    dab_tune(freq_index);
    Serial.print(F("\nScanning DAB Freq Index: "));
    sprintf(buffer, "%02u  %03u.", freq_index, (uint16_t)(dabfreq / 1000));
    Serial.print(buffer);
    sprintf(buffer, "%03u", (uint16_t)(dabfreq % 1000));
    Serial.print(buffer);
    Serial.print(F(" MHz"));
    if(dab_service_valid() == true)
    {
      dab_ensemble_info();
    }
  }
  Serial.print(F("\n"));
}

//-----------------------------------------------------------------------------------------
// Displays the tuned DAB frequency and service list
//-----------------------------------------------------------------------------------------
void dab_ensemble_info(void)
{
  char buffer[16];
  uint8_t i;

  Serial.print(F("\nEnsemble Freq Index:"));
  sprintf(buffer, "%02u  %03u.", freq_index, (uint16_t)(dabfreq / 1000));
  Serial.print(buffer);
  sprintf(buffer, "%03u", (uint16_t)(dabfreq % 1000));
  Serial.print(buffer);
  Serial.print(F(" MHz RSSI="));
  sprintf(buffer, "%d", rssi);
  Serial.print(buffer);
  Serial.print(F(" SNR="));
  sprintf(buffer, "%d\n", snr);
  Serial.print(buffer);
  Serial.print(ensemble_name);

  Serial.print(F("\nServices:\n"));
  Serial.print(F("ID\tName\n"));

  for (i = 0; i < numberofservices; i++)
  {
    Serial.print(i);
    Serial.print(F(":\t"));
    Serial.print(service[i].label);
    Serial.print(F("\n"));
  }
}

//-----------------------------------------------------------------------------------------
// test DAB function of DAB+ module
// return  sucessful true/false
//-----------------------------------------------------------------------------------------
bool dab_module_test(void)
{
	char buffer[16];

	dab_status = si468x_reset();
	if (dab_status != 0) {
		Serial.print(F(" Reset ERROR: "));
		sprintf(buffer, "%u\n", dab_status);
		Serial.print(buffer);    
	} else {
		Serial.print(F(" Reset PASS \n")); 
	
		dab_status = si468x_get_part_info();
		if (dab_status != 0) {
			Serial.print(F(" Get Part Info ERROR: "));
			sprintf(buffer, "%u\n", dab_status);
			Serial.print(buffer);    
		} else {
			Serial.print(F(" Get Part Info PASS -> Chip Revision:"));
			sprintf(buffer, "%u, RomID:%u, Part No:%u\n", chip_rev, rom_id, part_no);
			Serial.print(buffer);   

			dab_status = si468x_load_init();
			if (dab_status != 0) {
				Serial.print(F(" Load Init ERROR: "));
				sprintf(buffer, "%u\n", dab_status);
				Serial.print(buffer);    
			} else {
				Serial.print(F(" Load Init PASS \n")); 
			      
			  dab_status = si468x_host_load();
			  if (dab_status != 0) {
			    Serial.print(F(" Host Load ERROR: "));
					sprintf(buffer, "%u\n", dab_status);
					Serial.print(buffer);    
			  } else {
					Serial.print(F(" Host Load PASS \n")); 

					dab_status = si468x_load_init();
					if (dab_status != 0) {
						Serial.print(F(" Load Init ERROR: "));
			      sprintf(buffer, "%u\n", dab_status);
			      Serial.print(buffer);    
					} else {
			      Serial.print(F(" Load Init PASS \n")); 

						//Set SPI Clock to 10 MHz
						dab_status = si468x_flash_set_property(0x0001, 5000);
						if (dab_status != 0) {
							Serial.print(F(" FLASH Set Property ERROR: "));
							sprintf(buffer, "%u\n", dab_status);
							Serial.print(buffer);    
						} else {
							Serial.print(F(" FLASH Set Property PASS \n")); 

							dab_status = si468x_flash_load(FLASH_START_ADDR_DAB);  
							if (dab_status != 0) {
								Serial.print(F(" FLASH Load DAB ERROR: "));
								sprintf(buffer, "%u\n", dab_status);
								Serial.print(buffer);    
							} else {
								Serial.print(F(" FLASH Load DAB PASS \n"));  
								                      
								dab_status = si468x_boot(&value);
								if (dab_status != 0) {
									Serial.print(F(" Boot ERROR: "));
									sprintf(buffer, "%u -> Stat:%X\n", dab_status, value);
									Serial.print(buffer);    
								} else {
									Serial.print(F(" Boot PASS -> Stat:"));
								 	sprintf(buffer, "%X\n", value);
								  Serial.print(buffer);  

									dab_status = si468x_get_func_info();
									if (dab_status != 0) {
										Serial.print(F(" Get Func ERROR: "));
										sprintf(buffer, "%u\n", dab_status);
										Serial.print(buffer);    
							    } else {
							    	Serial.print(F(" Get Func PASS -> DAB Ver:"));
										sprintf(buffer, "%u.%u.%u\n", ver_major, ver_minor, ver_build);
										Serial.print(buffer);  

										dab_status = si468x_dab_set_freq_list();
										if (dab_status != 0) {
											Serial.print(F(" Set Freq list ERROR: "));
											sprintf(buffer, "%u\n", dab_status);
											Serial.print(buffer);    
										} else {
											Serial.print(F(" Set Freq list PASS \n"));   
							        
											//Set up INTB
											dab_status = si468x_set_property(0x0000, 0x0010);

											dab_status = si468x_set_property(0x1710, 0xF9FF);
											dab_status = si468x_set_property(0x1711, 0x0172);
											dab_status = si468x_set_property(0x1712, 0x0001);

											dab_status = si468x_set_property(0x8100, 0x0001);  //enable DSRVPCKTINT
											dab_status = si468x_set_property(0xb400, 0x0007);  //enable XPAD data
											if (dab_status != 0) {
												Serial.print(F(" Set Property ERROR: "));
												sprintf(buffer, "%u\n", dab_status);
												Serial.print(buffer);    
											} else {
												Serial.print(F(" Set Property PASS \n")); 
												return(true);						 
 											}
 										}
 									}
 								}
 							}
 						}
 					}
 				}
 			}
 		}
 	}
 	return(false);
}


  
