/*
 * TonOhmMeter.c
 *
 * Created: 8/23/2012 9:38:16 PM
 *  Author: Rolf Bulla
 *
 * Firmware for TonOhmMeter V1.0
 *
 * Version 1.1 
 *
 */ 
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <avr/wdt.h>
#include <util/delay.h>
#define F_CPU 8000000L

// Constants
#define TONE_MAX	1000	// above this limit ADC value it's an open (no tone) 
#define TONE_OFFSET 250		// base frequency added to ADC value (= 2KHz)
#define WAKEUPTONE	500		// Wake up tone at start up (= 1KHz)
#define CHARGE100	624		// LiPo voltage level for 100% (=3.70V -> 0.67V)
#define CHARGE75	596		// LiPo voltage level for 75%  (=3.56V -> 0.64V)
#define CHARGE50	577		// LiPo voltage level for 50%  (=3.45V -> 0.62V)
#define CHARGE25	559		// LiPo voltage level for 25%  (=3.34V -> 0.60V)
#define KEYTIME		16		// press button time count limit for power down in 0.125 sec ticks (=2 sec)
#define OFFTIME		2400	// time out count limit for power down in 0.125 sec ticks (=5 min)

#define WIPER0		0		// POT Wiper 0 register address
#define WTCON		4		// POT TCON register address
#define WSTATUS     5		// POT status register	
#define DEF_GAIN	10		// POT default gain level for stand-by

// Variables
int16_t i;				// as usual...
uint8_t count_1khz=0;	// counter to generate 1KHZ signal
uint8_t actmode=1;		// default mode setting
uint8_t prevmode=1;		// previous selected mode
uint8_t key_timer=0;	// wait counter for key off 
uint16_t off_timer=0;	// wait counter for time out
uint8_t set_sleep = 0;	// 1 = goto sleep mode
uint16_t adcres;		// ADC result of ohmmeter measurement
uint16_t tone;			// tone frequency setting
uint16_t gain[3] = {22,135,240};		// digital values for POT (range 1,2,3)

// ######### S u b r o u t i n e s #########

// set range 1..3 (0 for off)
void setmode(uint8_t mode){
	
	PORTB |= (1<<PB1) | (1<<PB0);
	PORTA |= (1<<PA3);	
	switch (mode)
	{
	case 0:
		PORTB &= ~(1<<PB1) & ~(1<<PB0);
		PORTA &= ~(1<<PA3);
		DDRA &= ~(1<<PA3);
		DDRB &= ~(1<<PB0) & ~(1<<PB1);
		SPI_write(WIPER0, DEF_GAIN);			
		break;
	case 1:
		DDRA |= (1<<PA3);
		DDRB |= (1<<PB0) | (1<<PB1);
		PORTB &= ~(1<<PB1) ;
		SPI_write(WIPER0, gain[0]);
		break;
	case 2:
		PORTB &= ~(1<<PB0) ;
		SPI_write(WIPER0, gain[1]);
		break;		
	case 3:
		PORTA &= ~(1<<PA3) ;
		SPI_write(WIPER0, gain[2]);
		break;
	}
}
	
// show battery charge level >25/>50/>75%	
uint8_t show_charge(void) {
	
	uint16_t vbatt=0;
	uint8_t	charge_ok=0;
	
	// check battery charge level
	ADMUX |= (1<<MUX0);			// select ADC1
	for (i=0;i<10;i++) {
		ADCSRA |= (1<<ADSC);	// start conversion
		while(ADCSRA & (1<<ADSC));	// wait until conversion is done
		vbatt += ADC;
	}	
	vbatt /= 10;				// average to reduce noise
	PORTB |= (1<<PB1) | (1<<PB0);
	PORTA |= (1<<PA3);	

	// 25% capacity
	if (vbatt > CHARGE25) {		
		PORTA &= ~(1<<PA3);	
		charge_ok++;
	}	
	// 50% capacity
	if (vbatt > CHARGE50) {
		PORTB &= ~(1<<PB0);			
		charge_ok++;
	}		
	// 75% capacity
	if (vbatt > CHARGE75) {
		PORTB &= ~(1<<PB1);		
		charge_ok++;
	}
	// no capacity (< 25%) --> charge_ok = 0
	ADMUX &= ~(1<<MUX0);		// disable ADC1
	return (charge_ok);
}

// condition device for operation
void init(void) {
	
	// set OC0B (PA7) to output 25KHz signal
	// set PA6 to output for 1KHz signal and SCK (POT)
	// set OC1B (PA5) to output for loudspeaker
	// Set PA4 to output SDI/SDO (POT)
	PORTA = (1<<PA7) | (1<<PA4) | (1<<PA5) | (1<<PA6);	
	DDRA = (1<<PA7) | (1<<PA4) | (1<<PA5) | (1<<PA6);
	
	// Set LED ports (PB0, PB1, PA3) to output
	// Set /CS for digital POT (PA2) to output
	PORTA |= (1<<PA3) | (1<<PA2);	
	DDRA |= (1<<PA3) | (1<<PA2);
	PORTB |= (1<<PB1) | (1<<PB0);
	DDRB = (1<<PB0) | (1<<PB1);
	
	// Set TIMER0 for output 25KHz (charge pump and derived signal 1KHz)
	TCCR0A = (1<<COM0B0) | (1<<WGM01);				// Toggle mode and CTC
	TCCR0B = (1<<WGM02) | (1<<CS01);				// CLK/8
	TIMSK0 = (1<<OCIE0B); 							// INT TIM0_COMPB enabled
	TIFR0 = (1<<OCF0B) | (1<<OCF0A) | (1<<TOV0);	// clear timer INT flags
	OCR0A = 20;
	OCR0B =	20;

	// Set TIMER1 for output 2KHz (loudspeaker)		
	TCCR1A = (1<<COM1B0);							// Toggle mode, normal operation
	TCCR1B = (1<<WGM12) | (1<<CS11);				// CLK/8
	TIMSK1 = (1<<OCIE1B);							// INT TIM1_COMPB enabled
	TIFR1 = (1<<OCF1B) | (1<<OCF1A) | (1<<TOV1);	// clear timer INT flags
	OCR1A = WAKEUPTONE;
	OCR1B =	WAKEUPTONE;	

	// Set ADC
	ADMUX = (1<<REFS1);								// Internal AVREF = 1.1V, select ADC0
	ADCSRA = (1<<ADEN) | (1<<ADSC);					// ADC on, start conversion, single trigger, INT disable
	ADCSRB = 0;										// free running mode
	DIDR0 = (1<<ADC1D) | (1<<ADC0D);				// Disable digital I/O at ADC0

	// Enable INT
	MCUCR = (1<<SM1) | (1<<ISC01);					// power down mode, falling edge INT0
	MCUCR |= (1<<BODS) | (1<<BODSE);				// disable brown-out detector 
	GIMSK = (1<<INT0);								// enable INT0
	set_sleep_mode(SLEEP_MODE_PWR_DOWN);			
	MCUSR = 0;										// clear all flags
	
	// Enable WDT 	
	wdt_disable();	
	WDTCSR = (1<<WDIF) | (1<<WDP1) | (1<<WDP0);		// CLK/16k = 0.125s
	WDTCSR |= (1<<WDIE) | (1<<WDE);					// enable interrupt
	
	// Power save
	PRR = (1<<PRUSI); 								// USI isn't used
	ACSR = (1<<ACD);								// Analog comparator as well
}

// SPI read emulation (with SDI/SDO multiplexing)
uint16_t SPI_read(uint8_t addr) {
uint16_t rval=0;

	TIMSK0 &= ~(1<<OCIE0B); 		// Stop 1KHz (PA6)
	TCCR1B = 0;						// stop timer (no tone signal)
	
	PORTA |= (1<<PA6) | (1<<PA4);	// SCK=SDIO HIGH
	DDRA |= (1<<PA6) | (1<<PA4);	// SCK+SDIO to output
	
	PORTA &= ~(1<<PA2);				// CS low
	
	// Address AD0..3 (bit15...12)
	for (i=3;i>=0;i--) {
		PORTA ^= (1<<PA6);			// SCK = LOW
		if ((addr & (1<<i)) == 0)
			PORTA &= ~(1<<PA4);
		else
			PORTA |= (1<<PA4);
			PORTA ^= (1<<PA6);			// SCK = HIGH
	}
	
	// Command C1,C0 (bit11,10)
	PORTA ^= (1<<PA6);				// SCK = LOW
	PORTA |= (1<<PA4);				// Data = HIGH
	_delay_us(5);
	PORTA ^= (1<<PA6);				// SCK = HIGH
	_delay_us(5);
	PORTA ^= (1<<PA6);				// SCK = LOW
	_delay_us(5);
	PORTA ^= (1<<PA6);				// SCK = HIGH
	
	// Data D9..0 (bit9..0)
	DDRA &= ~(1<<PA4);				// Set SDIO to input
	PORTA |= (1<<PA4);				// pull up on
	for (i=9;i>=0;i--) {
		PORTA ^= (1<<PA6);			// SCK = LOW
		_delay_us(5);
		PORTA ^= (1<<PA6);			// SCK = HIGH
		if ((PINA & (1<<PINA4)) != 0) {
			rval |= (1<<i);
		}
	}
	
	PORTA |= (1<<PA2);				// CS to HIGH
	DDRA |= (1<<PA4);				// SDIO to output
	
	TIMSK0 |= (1<<OCIE0B);			// Restart 1KHz (PA6)
	TCCR1B = (1<<WGM12) | (1<<CS11);	// Restart TIMER with CLK/8
	return(rval);
}

// SPI write emulation (with SDI/SDO multiplexing)	
void SPI_write(uint8_t addr, uint16_t data)  {
	TIMSK0 &= ~(1<<OCIE0B); 		// Stop 1KHz (PA6)
	TCCR1B = 0;						// stop timer (no tone signal)
	
	PORTA |= (1<<PA6) | (1<<PA4);	// SCK=SDIO HIGH
	DDRA |= (1<<PA6) | (1<<PA4);	// SCK+SDIO to output

	PORTA &= ~(1<<PA2);				// CS low
	
	// Address AD0..3 (bit15...12)
	for (i=3;i>=0;i--) {
		PORTA ^= (1<<PA6);			// SCK = LOW
		if ((addr & (1<<i)) == 0)
		PORTA &= ~(1<<PA4);
		else
		PORTA |= (1<<PA4);
		PORTA ^= (1<<PA6);			// SCK = HIGH
	}
	
	// Command C1,C0 (bit11,10)
	PORTA ^= (1<<PA6);				// SCK = LOW
	PORTA &= ~(1<<PA4);				// Data = LOW
	_delay_us(5);
	PORTA ^= (1<<PA6);				// SCK = HIGH
	_delay_us(5);
	PORTA ^= (1<<PA6);				// SCK = LOW
	_delay_us(5);
	PORTA ^= (1<<PA6);				// SCK = HIGH
	
	// Data D9..0 (bit9..0)
	DDRA |= (1<<PA4);				// set SDIO to output
	for (i=9;i>=0;i--) {
		PORTA ^= (1<<PA6);			// SCK = LOW
		if ((data & (1<<i)) != 0) {
			PORTA |= (1<<PA4);
		} else {
			PORTA &= ~(1<<PA4);
		}
		PORTA ^= (1<<PA6);			// SCK = HIGH
	}
	
	PORTA |= (1<<PA2);				// CS to HIGH
	DDRA |= (1<<PA4);				// SDIO to output
	
	TIMSK0 |= (1<<OCIE0B);			// Restart 1KHz (PA6)
	TCCR1B |= (1<<WGM12) | (1<<CS11);	// Restart TIMER with CLK/8
}

// trim gain level until ADC result = 1000
uint16_t trim_gain(void) {

uint16_t imin=1, imax=255;	
uint16_t imid;

	while (imax >= imin) {	
		
		imid = (imin+imax)/2;

		SPI_write(WIPER0, imid);
		_delay_us(20000);	
		
		adcres = 0;	
		for (i=0;i<50;i++) {
			ADCSRA |= (1<<ADSC);				// start conversion
			while(ADCSRA & (1<<ADSC));			// wait until conversion is done
			adcres += ADC;
		}
		adcres /= 50;							// average measurement to reduce noise

		// determine which subarray to search
		if      (adcres <  950)
		// change min index to search upper subarray
		imin = imid + 1;
		else if (adcres > 1000 )
		// change max index to search lower subarray
		imax = imid - 1;
		else
		// key found at index imid
		return imid;
	}
	// key not found
	return 0;
}


// ######### M A I N ##########
int main(void)
{	
	uint16_t spi_read;
	
	init();				// set device condition
	set_sleep = 0;		// no sleep mode yet
	
	// check battery charge level
	if (!show_charge()) set_sleep=1;
	_delay_ms(2000);
	
	setmode(actmode);
	sei();	
	tone=0;
	
	//gain[0]= trim_gain();
	//SPI_write(WIPER0, gain[0]);

	//spi_read = SPI_read(WIPER0);		
	//tone = spi_read;
		
	while(1)  {		
		
		// get the correlation voltage for R under test (RUT) 
		adcres = 0;
		for (i=0;i<20;i++) {
			ADCSRA |= (1<<ADSC);				// start conversion
			while(ADCSRA & (1<<ADSC));			// wait until conversion is done
			adcres += ADC;
		}
		adcres /= 20;							// average measurement to reduce noise
		
		// normal operation if no sleep request is pending
		if (!set_sleep) {
			if (adcres > TONE_MAX) {
				TCCR1B = 0;						// stop timer (no tone signal)
			} else {
				tone = adcres + TONE_OFFSET;
				TCCR1B = (1<<WGM12) | (1<<CS11);	// start timer for tone signal
				off_timer = 0;					// user still alive --> reset off timer
			}
		}
		
		// enter sleep if button isn't pressed any longer
		if (set_sleep==1 && (PINB & (1<<PINB2))) {
			MCUSR = 0;			
			setmode(0);
			wdt_disable();	
			cli();
			MCUCR = (1<<SM1);					// set INT0 level trigger	
			//PORTA |= (1<<PA6);					// set 1KHz signal line to high to save few mA
			PORTA &= ~(1<<PA6);					// set 1KHz signal line to low to save few mA
			ADCSRA &= ~(1<<ADEN);				// ADC off will save few uA
			sleep_enable();
			sei();
			sleep_cpu();
			sleep_disable();	
			cli();
			MCUCR = (1<<SM1) | (1<<ISC01);		// set INT0 falling edge trigger	
			ADCSRA |= (1<<ADEN);				// ADC on again
			set_sleep = 0;
			if (!show_charge()) set_sleep = 1;;
			OCR1A = WAKEUPTONE;
			OCR1B =	WAKEUPTONE;		
			TCCR1B = (1<<WGM12) | (1<<CS11);	// start Timer 1 for tone signal
			_delay_ms(2000);
			TCCR1B = 0;							// stop timer 
			actmode = 1;
			setmode(actmode);
			WDTCSR = (1<<WDIF) | (1<<WDP1) | (1<<WDP0);		// CLK/16k = 0.125s
			WDTCSR |= (1<<WDIE) | (1<<WDE);					// enable interrupt
			sei();			
		}
	}	
	return 0;		
}

// ######### I S R ##########
// tone interrupt
ISR(TIM1_COMPB_vect) {
	OCR1A = tone;			// update tone frequency
	OCR1B = tone;			
}

// Press button interrupt
ISR(INT0_vect) {
	actmode++;				// toggle modes
	if (actmode > 3) actmode=1;
	setmode(actmode);
	off_timer = 0;			// user still alive -> reset off timer
}

// fixed frequency interrupt
ISR(TIM0_COMPB_vect) {
	count_1khz++;
	// divider to scale down to 1KHz	
	if (count_1khz == 25) {
		PORTA ^= (1<<PA6);
		count_1khz = 0;
	}
}

// watchdog interrupt for power down time out
ISR(WDT_vect) {
	wdt_reset();
	WDTCSR |= (1<<WDIE) ;			// enable interrupt again
	if (PINB & (1<<PINB2)) key_timer = 0;
	key_timer++;					// count key timer
	off_timer++;					// count off timer
	// check if button is pressed for about 2 sec or timeout is reached	
	if ((key_timer > KEYTIME) || (off_timer > OFFTIME)) {
		// request sleep mode
		key_timer = 0;
		off_timer = 0;
		set_sleep = 1;
		setmode(0);
		TCCR1B = 0;					// stop timer (no tone signal)
	}
}