from tkinter import *
from tkinter import ttk
from tkinter import filedialog
import serial
import time
import sys
import glob
import sqlite3 as sl
import pandas
import csv
import matplotlib
from matplotlib.backends.backend_tkagg import (
    FigureCanvasTkAgg,
    NavigationToolbar2Tk
)
from matplotlib.figure import Figure
import matplotlib.animation as animation
matplotlib.use('TkAgg')


def bytes_to_hexstr(b: bytes) -> str:
    return "".join("{:02x}".format(byte) for byte in b)


multiply = [[0.01, 1, 1, 0.01, 1, 1, 0.1, 0.01, 0.001, 1, 0.01, 0.0001, 0.001, 0.1, 1, 0.01],
            [1, 0.0001, 0.0001, 1, 0.01, 0.001, 1, 0.1, 0.01, 0.001, 1, 1, 0.01, 1, 1, 1],
            [1, 0.001, 0.001, 1, 0.0001, 0.01, 1, 1, 1, 1, 1, 1, 0.0001, 1, 1, 1],
            [1, 0.01, 0.01, 1, 0.001, 0.0001, 1, 1, 1, 1, 1, 1, 0.001, 1, 1, 1],
            [1, 0.1, 0.1, 1, 0.01, 0.001, 1, 1, 1, 1, 1, 1, 0.01, 1, 1, 1],
            [1, 1, 1, 1, 0.0001, 0.01, 1, 1, 1, 1, 1, 1, 0.0001, 1, 1, 1],
            [1, 1, 1, 1, 0.001, 0.0001, 1, 1, 1, 1, 1, 1, 0.001, 1, 1, 1],
            [1, 1, 1, 1, 0.01, 0.001, 1, 1, 1, 1, 1, 1, 0.01, 1, 1, 1]]


class UT71:
    baud = 2400
    parity = serial.PARITY_ODD
    stop = serial.STOPBITS_ONE
    bits = serial.SEVENBITS
    strLine = ""
    ser_port = None
    record = FALSE
    conn_db = sl.connect('ut_meas.db')
    acdctxt = ""
    unittxt = ""
    autotxt = ""
    UT_value = 0
    filetypes = (
        ('CSV files', '*.csv'),
        ('All files', '*.*')
    )

    def __init__(self, master):
        self.master = master
        master.title("UT71 Measurement Tool")
        self.tk_status = StringVar()
        self.rec_cnt = IntVar()
        self.rec_interval = IntVar()
        self.rec_cnt.set(0)
        self.rec_interval.set(1)
        # Set timer
        self.counter = 0
        # setup db
        cur = self.conn_db.cursor()
        sql = 'CREATE TABLE IF NOT EXISTS ' \
              'Records (ID INTEGER PRIMARY KEY, SDATE TEXT, ACDC TEXT, VALUE REAL, UNIT TEXT, AUTO TEXT)'
        cur.execute(sql)
        cur.execute("BEGIN")  # start transaction
        n = cur.execute("SELECT COUNT() FROM Records").fetchone()[0]
        self.rec_cnt.set(n)
        self.draw_elements()
        self.plot_rec(0)
        root.after(50, self.serial_monitor)
        self.anim = animation.FuncAnimation(self.f, self.plot_rec, interval=1000)
        root.after(100, self.click_rec_stop)

    def draw_elements(self):

        # create all of the main containers
        top_frame = Frame(root, width=600, height=50)
        display_frame = Frame(root, width=600, height=100)
        record_frame = Frame(root, width=600, height=50)
        plot_frame = Frame(root, width=600, height=200)
        btm_frame = Frame(root, width=600, height=50)

        top_frame.grid(row=1, sticky="ew")
        display_frame.grid(row=2, sticky="ew")
        record_frame.grid(row=3, sticky="ew")
        plot_frame.grid(row=4, sticky="ew")
        btm_frame.grid(row=5, sticky="ew")

        # TOP FRAME
        Button(top_frame, image=ph_exit, command=self.client_exit).grid(row=0, column=0, pady=5)
        Label(top_frame, width=45, text="").grid(row=0, column=1)
        Button(top_frame, image=ph_usb, command=self.click_usb).grid(row=0, column=2)
        self.l_connect = Label(top_frame, text="—", width=1, bg='gray')
        self.l_connect.grid(row=0, column=3)

        # DISPLAY FRAME
        self.lbl_auto = Label(display_frame, width=4, text="AUTO")
        self.lbl_auto.grid(row=0, column=0)
        self.lbl_minus = Label(display_frame, text="-",font=("DSEG7 Classic Mini", 36),
              justify=LEFT, anchor='w', fg="white")
        self.lbl_minus.grid(row=1, column=0)
        self.lbl_acdc = Label(display_frame, text="AC+DC")
        self.lbl_acdc.grid(row=2, column=0)
        self.lbl_value = Label(display_frame, text="01.234", font=("DSEG7 Classic Mini", 96),
              justify=LEFT, anchor='w', fg="white")
        self.lbl_value.grid(row=0, column=1, rowspan=3)
        self.can_piep = Canvas(display_frame, height=32, width=32)
        self.stat_piep = self.can_piep.create_image(3, 3, image=ph_piep, anchor=NW)
        self.can_piep.grid(row=0, column=2)
        self.can_diode = Canvas(display_frame, height=32, width=32)
        self.can_diode.grid(row=1, column=2)
        self.stat_diode = self.can_diode.create_image(3, 3, image=ph_diode, anchor=NW)
        self.lbl_unit = Label(display_frame, width=4, text="UNIT", font=('Vedana', 24))
        self.lbl_unit.grid(row=2, column=2)
        self.analog = ttk.Progressbar(display_frame, orient='horizontal', mode='determinate', length=500, maximum=2)
        self.analog.grid(row=3, column=0, columnspan=8, padx=5, pady=10)

        # RECORD FRAME
        Button(record_frame, image=ph_start, command=self.click_rec_start).grid(row=0, column=0, rowspan=2)
        Button(record_frame, image=ph_stop, command=self.click_rec_stop).grid(row=0, column=1, rowspan=2, sticky='w')
        Label(record_frame, text="Records:").grid(row=0, column=2, sticky='e')
        Label(record_frame, textvariable=self.rec_cnt, justify=RIGHT, anchor='e').grid(row=0, column=3, sticky='w')
        Button(record_frame, image=ph_save, command=self.click_rec_save).grid(row=0, column=4, rowspan=2)
        Button(record_frame, image=ph_table, command=self.click_table).grid(row=0, column=5, rowspan=2)
        Button(record_frame, image=ph_delete, command=self.click_rec_delete).grid(row=0, column=6, rowspan=2)
        Label(record_frame, text="Interval [sec]:").grid(row=1, column=2, sticky='e')
        ent_interval = Entry(record_frame, justify=RIGHT, width=5, textvariable=self.rec_interval)
        ent_interval.grid(row=1, column=3, rowspan=2, sticky='w')
        Label(record_frame, text="", width=50).grid(row=3, column=0, columnspan=6, sticky='e')

        # PLOT FRAME
        self.f = Figure(figsize=(5, 3), dpi=100, frameon=FALSE, layout='tight')
        can_diagram = FigureCanvasTkAgg(self.f, plot_frame)
        self.a = self.f.add_subplot(1, 1, 1)
        toolbar = NavigationToolbar2Tk(can_diagram, plot_frame, pack_toolbar=False)
        can_diagram.get_tk_widget().grid(row=0, column=0)
        toolbar.grid(row=1, column=0)
        self.a.tick_params(axis='y', which='major', labelsize=8)
        self.a.tick_params(axis='x', which='major', labelsize=8)

        # BOTTOM FRAME
        Label(btm_frame, text="Error:").grid(row=0, column=0)
        Label(btm_frame, textvariable=self.tk_status, width=55, anchor=W, bg='gray').grid(row=1, column=0)
        self.tk_status.set("n/a")

    def close(self):
        pop.destroy()

    def click_usb(self):
        global pop
        pop = Toplevel(root)
        pop.title("USB-Serial Interface")
        pop.geometry("450x50")
        # Create a Label Text
        Label(pop, text="Serial I/F: ").grid(row=0, column=0)
        self.serial_list = self.populate_list()
        self.tk_choice_serial = StringVar()
        self.tk_choice_serial.set(self.serial_list[0])
        self.ifmenu = OptionMenu(pop, self.tk_choice_serial, *self.serial_list, command=self.option_select)
        self.ifmenu.grid(row=0, column=1, ipadx=3)
        self.ifmenu.config(width=30)
        # Add Button for making selection
        Button(pop, text="Rescan I/F", command=self.update_option_menu).grid(row=1, column=0)
        Button(pop, text="close", command=self.close).grid(row=1, column=1)

    def click_table(self):
        global pop
        pop = Toplevel(root)
        pop.title("Records")
        pop.resizable(width=1, height=1)
        frame = Frame(pop)
        frame.pack()
        bottomframe = Frame(pop)
        bottomframe.pack(side=BOTTOM)
        treev = ttk.Treeview(frame, selectmode='browse', height=20)
        treev.pack(side=LEFT, fill=BOTH)
        verscrlbar = ttk.Scrollbar(frame, orient="vertical", command=treev.yview)
        verscrlbar.pack(side=RIGHT, fill=Y)
        treev.configure(yscrollcommand=verscrlbar.set)
        treev["columns"] = ("1", "2", "3", "4", "5", "6")
        treev['show'] = 'headings'
        treev.column("1", width=50, anchor='c')
        treev.column("2", width=70, anchor='c')
        treev.column("3", width=50, anchor='c')
        treev.column("4", width=80, anchor='se')
        treev.column("5", width=50, anchor='c')
        treev.column("6", width=50, anchor='c')
        treev.heading("1", text="ID")
        treev.heading("2", text="TIME")
        treev.heading("3", text="ACDC")
        treev.heading("4", text="VALUE")
        treev.heading("5", text="UNIT")
        treev.heading("6", text="AUTO")
        cur = self.conn_db.cursor()
        cur.execute("SELECT * FROM Records")
        rows = cur.fetchall()
        for row in rows:
            treev.insert("", END, values=row)
        Button(bottomframe, text="Close", command=self.close).pack(side=BOTTOM)

    def click_rec_start(self):
        self.record = TRUE
        self.anim.resume()
        root.after(self.rec_interval.get()*1000, self.insert_rec)

    def click_rec_stop(self):
        self.record = FALSE
        self.anim.pause()

    def click_rec_delete(self):
        sql = 'DELETE FROM Records'
        cur = self.conn_db.cursor()
        cur.execute(sql)
        self.conn_db.commit()
        self.rec_cnt.set(0)
        self.plot_rec(0)

    def plot_rec(self, i):
        self.a.clear()
        sql = """SELECT ID, VALUE FROM Records"""
        data = pandas.read_sql(sql, self.conn_db)
        self.a.plot(data.ID, data.VALUE, color="#0a82ff", linewidth=1)

    def insert_rec(self):
        if self.record and self.ser_port:
            sql = 'INSERT INTO Records (SDATE, ACDC, VALUE, UNIT, AUTO) values(?, ?, ?, ?, ?)'
            strtime = time.strftime("%H:%M:%S", time.gmtime())
            data = [strtime, self.acdctxt, self.UT_value, self.unittxt, self.autotxt]
            cur = self.conn_db.cursor()
            cur.execute(sql, data)
            self.conn_db.commit()
            self.rec_cnt.set(self.rec_cnt.get() + 1)
            root.after(self.rec_interval.get()*1000, self.insert_rec)

    def click_rec_save(self):
        if self.rec_cnt.get() > 0:
            filename = filedialog.asksaveasfilename(title='Open a file', initialdir='.', filetypes=self.filetypes)
            if filename != "":
                cur = self.conn_db.cursor()
                cur.execute("SELECT * FROM Records;")
                with open(filename, 'w', newline='') as csv_file:
                    csv_writer = csv.writer(csv_file)
                    csv_writer.writerow([i[0] for i in cur.description])
                    csv_writer.writerows(cur)

    def update_option_menu(self):
        self.serial_list = self.populate_list()
        menu = self.ifmenu["menu"]
        menu.delete(0, "end")
        for string in self.serial_list:
            menu.add_command(label=string,
                             command=lambda value=string: self.tk_choice_serial.set(value))

    def option_select(self, *args):
        try:
            port = self.tk_choice_serial.get()
            if self.ser_port:
                self.ser_port.close()
                self.ser_port = None
                self.l_connect.config(bg='gray')
            if port != "---":
                self.ser_port = serial.Serial(port, self.baud, parity=self.parity, stopbits=self.stop,
                                              bytesize=self.bits, timeout=1)
                if not self.ser_port:
                    self.tk_status.set("ERROR: Couldn't open a serial port: " + port)
                else:
                    self.l_connect.config(bg='#0a82ff')
            else:
                if self.ser_port:
                    self.ser_port.close()
                    self.l_connect.config(bg='gray')
        except IOError as e:
            self.tk_status.set("Socket error({0}): {1}".format(e.errno, e.strerror))

    def populate_list(self):
        # Lists serial port names
        #   :raises EnvironmentError:
        #        On unsupported or unknown platforms
        #   :returns:
        #        A list of the serial ports available on the system
        if sys.platform.startswith('win'):
            ports = ['COM%s' % (i + 1) for i in range(256)]
        elif sys.platform.startswith('linux') or sys.platform.startswith('cygwin'):
            # this excludes your current terminal "/dev/tty"
            ports = glob.glob('/dev/tty[A-Za-z]*')
        elif sys.platform.startswith('darwin'):
            ports = glob.glob('/dev/tty.*')
        else:
            raise EnvironmentError('Unsupported platform')
        ports.insert(0, '---')
        return ports

    def client_exit(self):
        if self.ser_port:
            self.ser_port.close()
        self.conn_db.close()
        exit(0)

    def serial_monitor(self):
        root.after(50, self.serial_monitor)
        if self.ser_port and self.ser_port.in_waiting > 0:
            data = self.ser_port.read_until('\r\n', 11)
            self.decode(data)
            # print("RECEIVED: {}".format(bytes_to_hexstr(data)))

    # Decode 11 bytes from UT71A/B/C
    # ==============================
    # Byte / Bit          6    5    4    3    2    1    0
    # [0]    Digit 1      0    1    1    =====Digit======
    # [1]    Digit 2      0    1    1    =====Digit======
    # [2]    Digit 3      0    1    1    =====Digit======
    # [3]    Digit 4      0    1    1    =====Digit======
    # [4]    Digit 5      0    1    1    =====Digit======
    # [5]    Range        0    1    1    0    =see below=
    # [6]    Unit         0    1    1    ====see below===
    # [7]    Coupling     0    1    1    0    0    DC   AC    (also DC and AC possible)
    # [8]    Info         0    1    1    0    NEG  MAN  AUTO  (MAN or AUTO only)
    # [9]    '\r'         0    0    0    1    1    0    1
    # [10]   '\n'         0    0    0    1    0    1    0
    #
    # Digit: 0x30..0x39 = '0..9', 0x3A = ' ', 0x3B = '-', 0x3C = 'L', 0x3F = 'H'
    # REL not sent
    # No tansmission in HOLD state
    # If NEG set at Range 15 sent value is duty cycle
    # No LowBat info sent
    # Storaged data not accessable
    #
    #       Range:   0   1     2     3     4     5    6     7     8     9    10    11    12    13   14  15
    # Multiply:      mV  V     V     mV    Ω     F    °C    µA    mA    A    Pieps Diode Hz    °F   W   %
    #         '0'    .01 -     -     .01  -      -    .1   .01   .001   -    .01   .0001 .001  .1   -   .01
    #         '1'    -   .0001 .0001 -    .01   .001  -    .1    .01    .001 -     -     .01   -    -   -
    #         '2'    -   .001  .001  -    .0001 .01   -    -     -      -    -     -     .0001 -    -   -
    #         '3'    -   .01   .01   -    .001  .0001 -    -     -      -    -     -     .001  -    -   -
    #         '4'    -   .1    .1    -    .01   .001  -    -     -      -    -     -     .01   -    -   -
    #         '5'    -   -     -     -    .0001 .01   -    -     -      -    -     -     .0001 -    -   -
    #         '6'    -   -     -     -    .001  .0001 -    -     -      -    -     -     .001  -    -   -
    #         '7'    -   -     -     -    .01   .001  -    -     -      -    -     -     .01   -    -   -
    def decode(self, data):
        # range
        UT_range = data[5] & 0x07

        # unit
        UT_unit = data[6] & 0x0F
        UT_multiply = multiply[UT_range][UT_unit]

        # sign
        if (data[8] & 0x4) > 0:
            valuetxt = '-'
            UT_sign = -1
        else:
            valuetxt = ' '
            UT_sign = 1
        self.lbl_minus.config(text=valuetxt)

        # AUTO / MAN
        if (data[8] & 0x03) == 0x1:
            self.autotxt = 'AUTO'
        if (data[8] & 0x03) == 0x2:
            self.autotxt = 'MAN'
        self.lbl_auto.config(text=self.autotxt)

        # value
        valuetxt = ""
        self.UT_value = 0
        for i in range(0, 5):
            if data[i] == 0x3A:
                valuetxt += '    '
            elif data[i] == 0x3B:
                valuetxt += '-'
            elif data[i] == 0x3C:
                valuetxt += 'L'
            elif data[i] == 0x3F:
                valuetxt += 'H'
            else:
                valuetxt += chr(data[i])
                self.UT_value += (data[i] & 0xF) * 10**(4-i)
                if 10**-(4-i) == UT_multiply:
                    valuetxt += '.'
        self.lbl_value.config(text=valuetxt)
        # sign has different meaning for frequency measurement
        if self.UT_value != 12:
            self.UT_value = round(self.UT_value * UT_multiply * UT_sign, 4)
        else:
            self.UT_value = round(self.UT_value * UT_multiply, 4)

        # coupling
        if data[7] == 0x31:
            self.acdctxt = 'AC'
        elif data[7] == 0x32:
            self.acdctxt = 'DC'
        elif data[7] == 0x33:
            self.acdctxt = 'AC+DC'
        else:
            self.acdctxt = ""
        self.lbl_acdc.config(text=self.acdctxt)

        # unit
        if UT_unit == 0x0:
            valuetxt = 'mV~'
        if UT_unit == 0x1:
            valuetxt = 'V='
        if UT_unit == 0x2:
            valuetxt = 'V~'
        if UT_unit == 0x3:
            valuetxt = 'mV='
        if UT_unit == 0x4:
            if UT_range == 0x00 or UT_range == 0x01:
                valuetxt = 'Ω'
            elif UT_range == 0x02 or UT_range == 0x03 or UT_range == 0x04:
                valuetxt = 'kΩ'
            elif UT_range == 0x05 or UT_range == 0x06 or UT_range == 0x07:
                valuetxt = 'MΩ'
            else:
                valuetxt = ''
        if UT_unit == 0x5:
            if UT_range == 0x00 or UT_range == 0x01 or UT_range == 0x02:
                valuetxt = 'nF'
            elif UT_range == 0x03 or UT_range == 0x04 or UT_range == 0x05:
                valuetxt = 'µF'
            elif UT_range == 0x06 or UT_range == 0x07:
                valuetxt = 'mF'
            else:
                valuetxt = ''
        if UT_unit == 0x6:
            valuetxt = '°C'
        if UT_unit == 0x7:
            valuetxt = 'µA'
        if UT_unit == 0x8:
            valuetxt = 'mA'
        if UT_unit == 0x9:
            valuetxt = 'A'
        if UT_unit == 0xA:
            valuetxt = 'Ω'
        if UT_unit == 0xB:
            valuetxt = 'V'
        if UT_unit == 0xC:
            if UT_sign == -1:
                valuetxt = '%'
            else:
                if UT_range == 0x00 or UT_range == 0x01:
                    valuetxt = 'Hz'
                elif UT_range == 0x02 or UT_range == 0x03 or UT_range == 0x04:
                    valuetxt = 'KHz'
                elif UT_range == 0x05 or UT_range == 0x06 or UT_range == 0x07:
                    valuetxt = 'MHz'
                else:
                    valuetxt = ''
        if UT_unit == 0xD:
            valuetxt = '°F'
        if UT_unit == 0xE:
            valuetxt = 'W'
        if UT_unit == 0xF:
            valuetxt = '%'
        self.lbl_unit.config(text=valuetxt)
        self.unittxt = valuetxt

        # Symbol in case of piep and diode test
        if UT_unit == 10:
            self.can_piep.itemconfig(self.stat_piep, state='normal')
            self.can_diode.itemconfig(self.stat_diode, state='hidden')
        elif UT_unit == 11:
            self.can_piep.itemconfig(self.stat_piep, state='hidden')
            self.can_diode.itemconfig(self.stat_diode, state='normal')
        else:
            self.can_piep.itemconfig(self.stat_piep, state='hidden')
            self.can_diode.itemconfig(self.stat_diode, state='hidden')

        # analog bar
        self.analog['value'] = abs(self.UT_value)
        # print("Value:", UT_value, UT_unit, UT_range, UT_multiply)


root = Tk()
root.geometry("510x670")
root.resizable(False, False)
ph_exit = PhotoImage(file="exit_32.png")
ph_usb = PhotoImage(file="usb_32.png")
ph_diode = PhotoImage(file="diode_32.png")
ph_piep = PhotoImage(file="piep_32.png")
ph_start = PhotoImage(file="start_32.png")
ph_stop = PhotoImage(file="stop_32.png")
ph_save = PhotoImage(file="save_32.png")
ph_table = PhotoImage(file="table_32.png")
ph_delete = PhotoImage(file="delete_32.png")
ph_plus = PhotoImage(file="plus_32.png")
ph_fit = PhotoImage(file="fit_32.png")
ph_minus = PhotoImage(file="minus_32.png")
ph_zoom = PhotoImage(file="zoom_in_32.png")
ph_rewind2 = PhotoImage(file="rewind2_32.png")
ph_rewind = PhotoImage(file="rewind_32.png")
ph_forward = PhotoImage(file="forward_32.png")
ph_forward2 = PhotoImage(file="forward2_32.png")
app = UT71(root)
root.mainloop()
